create or replace package body tbicds.PCK_FX_SEC is

/* Copyright 2015 Intellica Corporation 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at
    http://www.apache.org/licenses/LICENSE-2.0
Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

/*audits a transaction to the syslog*/
  procedure SysLogAuditTransaction(pi_vSessionID in varchar2,
                                   pi_vSessionClientIP in varchar2,
                                   pi_nUserID          in number,
                                   pi_nStatus          in number,
                                   pi_vAuditXML        in varchar2,
                                   po_nStatusCode      out number,
                                   po_vStatusComment   out varchar2)
  
  is
  
  begin
    
    --6 = normal operational messages
    --3 = error conditions
  
    --default status to good
    po_nStatusCode    := 0; --0 = success, other than zero = failed
    po_vStatusComment := '';
  
  
    --commented out on drdb_dev since we are using windows DB
   -- if pi_nStatus != 0 then
   --    sys.writesyslog(3, pi_vAuditXML);
   -- else
    --   sys.writesyslog(6, pi_vAuditXML);
   -- end if;
    
   -- po_nStatusCode    := 1;
   -- po_vStatusComment := '167 - An error occurred while auditing a transaction, Please contact your system administrator.';
    
    return;
       
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '167 - An error occurred while auditing a transaction, Please contact your system administrator.';
  end;

  /*
  creates a database session for a user. used after a successful login to an external
  system such as MDWS to establish a session for the user in the database
  */
  procedure CreateSession(pi_vSessionID       in varchar2,
                          pi_vSessionClientIP in varchar2,
                          pi_nUserID          in number,
                          pi_vWebSessionID    in varchar2,
                          po_vDBSessionID     out varchar2,
                          po_nStatusCode      out number,
                          po_vStatusComment   out varchar2) 
   is
    v_vSql         varchar2(4000);
    v_vDBSessionID varchar2(4000) := '';
    v_vStat  varchar2(4000) := '';
    v_nStat number := 0;
  begin
    --get a new db session id:
    v_vDBSessionID := '';
    v_vSql         := 'select t.SID || :P0 || t.SERIAL# || t.USER# || :P1 || :P2 ' ||
                      'from sys.v_$session t ' || 'where t.audsid = :P3 ';
    execute immediate v_vSql
      into v_vDBSessionID
      using DBMS_CRYPTO.RandomBytes(3), DBMS_CRYPTO.RandomBytes(3), userenv('sessionid'), userenv('sessionid');
  
    --clean up any old sessions, users can only login once.
    v_vSql := 'delete from tbicds.fx_session where fx_user_id = :P0';
    execute immediate v_vSql
      using pi_nUserID;
    commit;
  
    v_vSql := 'delete from tbicds.fx_session where db_session_id = :P0';
    execute immediate v_vSql
      using v_vDBSessionID;
    commit;
  
    v_vSql := 'delete from tbicds.fx_session_value where db_session_id = :P0';
    execute immediate v_vSql
      using v_vDBSessionID;
    commit;
  
    --insert a record into fx_session
    v_vSql := 'insert into tbicds.fx_session (' ||
              'fx_user_id, web_session_id, db_session_id, date_created, date_last_action, expired, client_ip) ' ||
              'values (' || ':P0, :P1, :P2, :P3, :P4, :P5, :P6) ';
    execute immediate v_vSql
      using pi_nUserID, pi_vWebSessionID, v_vDBSessionID, sysdate, sysdate, 0, pi_vSessionClientIP;
  
    --add an audit record for the success
    --v_vSql := 'insert into tbicds.fx_audit (' ||
    --          'db_session_id, client_ip, fx_user_id, audit_date, audit_name, audit_data) ' ||
    --          'values (' || ':P0, :P1, :P2, :P3, :P4, :P5) ';
    --execute immediate v_vSql
    --  using v_vDBSessionID, pi_vSessionClientIP, pi_nUserID, sysdate, 'LOGIN', 'SUCCESS';
    --commit;
    begin
      
      SysLogAuditTransaction(v_vDBSessionID,
                             pi_vSessionClientIP,
                             pi_nUserID,
                             0,
                             'SESSION_CREATED' 
                             || '|' || pi_vSessionClientIP
                             || '|' || pi_nUserID
                             || '|' || 'SUCCESS',
                             v_nStat,
                             v_vStat);
   
    exception
    when others then
      null;
    end;
    
  
    --keep the session id
    po_vDBSessionID := v_vDBSessionID;
  
  exception
    when others then
      rollback;
      po_vDBSessionID   := '';
      po_nStatusCode    := pck_common.c_nStatus_Error;
      po_vStatusComment := '';
  end;

  /*validate the password against rules*/
  procedure ValidatePassword(pi_vKey           in varchar2,
                             pi_nUserID        in number,
                             pi_vUserName      in varchar2,
                             pi_vOldPassword   in varchar2,
                             pi_vPassword      in varchar2,
                             pi_vCOldPassword  in varchar2,
                             pi_vCPassword     in varchar2,
                             pi_vCUserName     in varchar2,
                             pi_nResetPassword in number,
                             po_nStatusCode    out number,
                             po_vStatusComment out varchar2) 
   is
  
    v_nFXUserID            number := 0;
    v_nUserCount           number := 0;
    v_dDatePasswordChanged date;
    v_vPREV_PWD1           varchar2(2000);
    v_vPREV_PWD2           varchar2(2000);
    v_vPREV_PWD3           varchar2(2000);
    v_vPREV_PWD4           varchar2(2000);
    v_vPREV_PWD5           varchar2(2000);
    v_vPREV_PWD6           varchar2(2000);
    v_vPREV_PWD7           varchar2(2000);
    v_vPREV_PWD8           varchar2(2000);
    v_vPREV_PWD9           varchar2(2000);
    v_vPREV_PWD10          varchar2(2000);
  
    --suat user personal information
    v_vUsrPDataName  varchar2(2000);
    v_vUsrPDataName1 varchar2(2000);
    v_vUsrPDataName2 varchar2(2000);
  
    v_vUsrPDataTitle    varchar2(2000);
    v_vUsrPDataUnit     varchar2(2000);
    v_vUsrPDataSquadron varchar2(2000);
    v_vUsrPDataPhone    varchar2(2000);
  
    --patient personal information
    v_vPatPDataFirstName  varchar2(2000);
    v_vPatPDataLastName   varchar2(2000);
    v_vPatPDataSSN        varchar2(2000);
    v_vPatPDataCity       varchar2(2000);
    v_vPatPDataPostalCode varchar2(2000);
    v_vPatPDataHomePhone  varchar2(2000);
    v_vPatPDataWorkPhone  varchar2(2000);
    v_dtPatPDataDOB       date;
    v_vPWDCheck           varchar2(4000);
  
    v_vCUserName1 varchar2(2000);
    v_vCUsername2 varchar2(2000);
    v_vUNCheck    varchar2(2000);
    v_nResetPWD   number;
  
    v_nPWDDiffCharacterCount number := 0;
    v_nPWDGreatest           number := 0;
    v_nPWDLeast              number := 0;
  
  begin
  
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    --get data needed for the check
    select date_password_changed,
           PREV_PWD1,
           PREV_PWD2,
           PREV_PWD3,
           PREV_PWD4,
           PREV_PWD5,
           PREV_PWD6,
           PREV_PWD7,
           PREV_PWD8,
           PREV_PWD9,
           PREV_PWD10,
           nvl(t.reset_password, 0)
      into v_dDatePasswordChanged,
           v_vPREV_PWD1,
           v_vPREV_PWD2,
           v_vPREV_PWD3,
           v_vPREV_PWD4,
           v_vPREV_PWD5,
           v_vPREV_PWD6,
           v_vPREV_PWD7,
           v_vPREV_PWD8,
           v_vPREV_PWD9,
           v_vPREV_PWD10,
           v_nResetPWD
      from tbicds.fx_user t
     where t.fx_user_id = pi_nUserID;
  
    --limit user changes to their account passwords 
    --once every 24 hours with the exception of privileged 
    --or administrative users. or in the case where the 
    --pwd was reset this keeps the user from spinning through pwds to get to
    --an old pwd for convenience
    if nvl(pi_nResetPassword, 0) < 1 then
      if (v_dDatePasswordChanged is not null) and
         (sysdate - 1 < v_dDatePasswordChanged) and
         not HASUSERRIGHT(v_nFXUserID, PCK_COMMON.c_nAdministratorUR) then
        po_nStatusCode    := 1;
        po_vStatusComment := 'Password cannot be changed in less than 24 hours since last change!';
        return;
      end if;
    end if;
  
    --make sure the password is not the same as the 10 past passwords
    if pi_nUserID > 0 then
    
      if pi_vPassword = v_vPREV_PWD1 or pi_vPassword = v_vPREV_PWD2 or
         pi_vPassword = v_vPREV_PWD3 or pi_vPassword = v_vPREV_PWD4 or
         pi_vPassword = v_vPREV_PWD5 or pi_vPassword = v_vPREV_PWD6 or
         pi_vPassword = v_vPREV_PWD7 or pi_vPassword = v_vPREV_PWD8 or
         pi_vPassword = v_vPREV_PWD9 or pi_vPassword = v_vPREV_PWD10 or
         pi_vPassword = pi_vOldPassword then
      
        po_nStatusCode    := 1;
        po_vStatusComment := 'New password cannot be the same as recent past password. Please choose a different password!';
        return;
      end if;
    
    end if;
  
    --ensure passwords do not contain personal information such as names, telephone numbers, account names, birthdates, or dictionary words. 
    for rec in (select t.restricted_name from fx_restricted_name t) loop
      if (instr(upper(pi_vCPassword), upper(rec.restricted_name)) > 0) then
        po_nStatusCode    := 1;
        po_vStatusComment := 'Password cannot contain personal information such as names, telephone numbers, account names, birthdates, or dictionary words!';
        return;
      end if;
    end loop;
  
    --new account passwords differ from the previous password by at least four characters when a password is changed. 
    select greatest(length(pi_vCOldPassword), length(pi_vCPassword)),
           least(length(pi_vCOldPassword), length(pi_vCPassword))
      into v_nPWDGreatest, v_nPWDLeast
      from dual;
    v_nPWDDiffCharacterCount := v_nPWDGreatest - v_nPWDLeast;
    if v_nPWDDiffCharacterCount < 4 then
      for i in 1 .. v_nPWDLeast loop
        if (substr(pi_vCOldPassword, i, 1) <> substr(pi_vCPassword, i, 1)) then
          v_nPWDDiffCharacterCount := v_nPWDDiffCharacterCount + 1;
        end if;
        -- exit stop_loop when v_nPWDDiffCharacterCount > 3;
      end loop; -- stop_loop;
      if v_nPWDDiffCharacterCount < 4 then
        po_nStatusCode    := 1;
        po_vStatusComment := 'Passwords must be different from the previous password by at least four characters!';
        return;
      end if;
    end if;
    --for i in 1..(length(pi_vCPassword) - 4)
    --loop
    --   if (instr(pi_vCOldPassword, substr(pi_vCPassword, i, 4)) > 0)
    --  then
    --      po_nStatusCode := 1;
    --      po_vStatusComment := 'Passwords must be different from the previous password by at least four characters!';
    --      return;
    --   end if;
    --end loop;
  
    v_vPWDCheck   := upper(pi_vCPassword);
    v_vUNCheck    := upper(pi_vCUserName);
    v_vCUserName1 := pck_common.GetPiece(v_vUNCheck, '.', 0);
    v_vCUserName2 := pck_common.GetPiece(v_vUNCheck, '.', 1);
    if instr(v_vPWDCheck, pi_vCUserName) > 0 or
       instr(v_vPWDCheck, v_vCUserName1) > 0 or
       instr(v_vPWDCheck, v_vCUserName2) > 0 then
    
      po_nStatusCode    := 1;
      po_vStatusComment := 'Password cannot contain elements of your user name!';
      return;
    end if;
  
    --make sure the password does not contain personal information
    --suat user personal information
    begin
      select name, Title, Unit, Squadron, Phone
        into v_vUsrPDataName,
             v_vUsrPDataTitle,
             v_vUsrPDataUnit,
             v_vUsrPDataSquadron,
             v_vUsrPDataPhone
        from tbicds.app_user
       where fx_user_id = v_nFXUserID;
    
      v_vUsrPDataName     := upper(v_vUsrPDataName);
      v_vUsrPDataTitle    := upper(v_vUsrPDataTitle);
      v_vUsrPDataUnit     := upper(v_vUsrPDataUnit);
      v_vUsrPDataSquadron := upper(v_vUsrPDataSquadron);
      v_vUsrPDataPhone    := upper(v_vUsrPDataPhone);
    
      v_vUsrPDataName1 := pck_common.GetPiece(v_vUsrPDataName, ' ', 0);
      v_vUsrPDataName2 := pck_common.GetPiece(v_vUsrPDataName, ' ', 1);
    
      if instr(v_vPWDCheck, v_vUsrPDataName) > 0 or
         instr(v_vPWDCheck, v_vUsrPDataName1) > 0 or
         instr(v_vPWDCheck, v_vUsrPDataName2) > 0 or
         instr(v_vPWDCheck, v_vUsrPDataTitle) > 0 or
         instr(v_vPWDCheck, v_vUsrPDataUnit) > 0 or
         instr(v_vPWDCheck, v_vUsrPDataSquadron) > 0 or
         instr(v_vPWDCheck, v_vUsrPDataPhone) > 0 or
         instr(v_vPWDCheck, replace(v_vUsrPDataPhone, '-', '')) > 0 then
      
        po_nStatusCode    := 1;
        po_vStatusComment := 'Password cannot contain personal information such as names, telephone numbers, account names, birthdates, or dictionary words!';
        return;
      
      end if;
    
    exception
      when others then
        null;
    end;
  
    --patient personal information
    begin
    
      select tbicds.fnc_utl_decstr(t.FIRST_NAME, pi_vKey, t.PATIENT_ID) as first_name,
             tbicds.fnc_utl_decstr(t.LAST_NAME, pi_vKey, t.PATIENT_ID) as last_name,
             tbicds.fnc_utl_decstr(t.SSN, pi_vKey, t.PATIENT_ID) as ssn,
             t.City,
             t.Postal_Code,
             t.HomePhone,
             t.WorkPhone,
             to_date(tbicds.fnc_utl_decstr(t.dob, pi_vKey, t.PATIENT_ID),
                     'MM/DD/YYYY') as dob
        into v_vPatPDataFirstName,
             v_vPatPDataLastName,
             v_vPatPDataSSN,
             v_vPatPDataCity,
             v_vPatPDataPostalCode,
             v_vPatPDataHomePhone,
             v_vPatPDataWorkPhone,
             v_dtPatPDataDOB
        from tbicds.patient_demographics t
       where t.fx_user_id = v_nFXUserID;
    
      if instr(v_vPWDCheck, v_vPatPDataFirstName) > 0 or
         instr(v_vPWDCheck, v_vPatPDataLastName) > 0 or
         instr(v_vPWDCheck, v_vPatPDataSSN) > 0 or
         instr(v_vPWDCheck, replace(v_vPatPDataSSN, '-', '')) > 0 or
        
         instr(v_vPWDCheck, v_vPatPDataCity) > 0 or
         instr(v_vPWDCheck, v_vPatPDataPostalCode) > 0 or
         instr(v_vPWDCheck, v_vPatPDataHomePhone) > 0 or
         instr(v_vPWDCheck, replace(v_vPatPDataHomePhone, '-', '')) > 0 or
        
         instr(v_vPWDCheck, v_vPatPDataWorkPhone) > 0 or
         instr(pi_vCPassword, to_char(v_dtPatPDataDOB, 'mmddyyyy')) > 0 or
         instr(pi_vCPassword, to_char(v_dtPatPDataDOB, 'ddmmyyyy')) > 0 or
         instr(pi_vCPassword, to_char(v_dtPatPDataDOB, 'mmddyy')) > 0 or
         instr(pi_vCPassword, to_char(v_dtPatPDataDOB, 'ddmmyy')) > 0 
         
         then
      
        po_nStatusCode    := 1;
        po_vStatusComment := 'Password cannot contain personal information such as names, telephone numbers, account names, birthdates, or dictionary words!';
        return;
      
      end if;
    exception
      when others then
        null;
    end;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '5. Invalid User Name/Password';
      return;
  end;

  /*delete all session values*/
  procedure DeleteAllSessionValues(pi_vSessionID       in varchar2,
                                   pi_vSessionClientIP in varchar2,
                                   pi_nUserID          in number,
                                   po_nStatusCode      out number,
                                   po_vStatusComment   out varchar2)
   is
  
    v_nCount number;
  
  begin
  
    --default status to good
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    v_nCount := 0;
  
    delete from tbicds.fx_session where db_session_id = pi_vSessionID;
    commit;
    delete from tbicds.fx_session_value
     where db_session_id = pi_vSessionID;
    commit;
    --delete from tbicds.patient_lock where session_id = pi_vSessionID;
    commit;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '158 - An error occurred while removing session values, Please contact your system administrator.';
  end;

  /*delete 1 session value*/
  procedure DeleteSessionValue(pi_vSessionID       in varchar2,
                               pi_vSessionClientIP in varchar2,
                               pi_nUserID          in number,
                               pi_vKey             in varchar2,
                               po_nStatusCode      out number,
                               po_vStatusComment   out varchar2) 
 is
  
    v_nCount number;
  
  begin
  
    --default status to good
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    v_nCount := 0;
  
    --check that there is a session matching this session id and client ip
    select count(*)
      into v_nCount
      from tbicds.fx_session
     where db_session_id = pi_vSessionID
       and client_ip = pi_vSessionClientIP;
  
    --scrap it all, someone has a mismatched ip/sessid combo
    --this will affewctivly log the user out
    if v_nCount < 1 then
    
      delete from tbicds.fx_session where db_session_id = pi_vSessionID;
      commit;
      delete from tbicds.fx_session_value
       where db_session_id = pi_vSessionID;
      commit;
    
      po_nStatusCode    := 1;
      po_vStatusComment := 'GetSessionValue: failed!';
      return;
    
    end if;
  
    delete from tbicds.fx_session_value
     where db_session_id = pi_vSessionID
       and client_ip = pi_vSessionClientIP
       and upper(db_session_key) = upper(pi_vKey);
    commit;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '159 - An error occurred while removing a session value, Please contact your system administrator.';
  end;

  /*get a session value*/
  procedure GetSessionValue(pi_vDBSessionID     in varchar2,
                            pi_vWebSessionID    in varchar2,
                            pi_vSessionClientIP in varchar2,
                            pi_nUserID          in number,
                            pi_vKey             in varchar2,
                            po_vKeyValue        out varchar2,
                            po_nStatusCode      out number,
                            po_vStatusComment   out varchar2) 
  is
  
    v_nCount number;
  
  begin
  
    --default status to good
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
    po_vKeyValue      := '';
  
    v_nCount := 0;
  
    --check that there is a session matching this session id and client ip
    select count(*)
      into v_nCount
      from tbicds.fx_session
     where db_session_id = pi_vDBSessionID
       and web_session_id = pi_vWebSessionID
       and client_ip = pi_vSessionClientIP;
  
    --scrap it all, someone has a mismatched ip/sessid combo
    --this will happen if the user times out for example
    --this will affewctivly log the user out
    if v_nCount < 1 then
    
      delete from tbicds.fx_session where db_session_id = pi_vDBSessionID;
      commit;
    
      delete from tbicds.fx_session_value
       where db_session_id = pi_vDBSessionID;
      commit;
    
      po_nStatusCode    := 1;
      po_vStatusComment := 'GetSessionValue failed, user may have timed out!';
      return;
    
    end if;
  
    select nvl(db_session_value, '')
      into po_vKeyValue
      from tbicds.fx_session_value
     where db_session_id = pi_vDBSessionID
       and client_ip = pi_vSessionClientIP
       and upper(db_session_key) = upper(pi_vKey);
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '160 - An error occurred while retrieving a session value, Please contact your system administrator.';
  end;

  /*set a session value*/
  procedure SetSessionValue(pi_vSessionID       in varchar2,
                            pi_vSessionClientIP in varchar2,
                            pi_nUserID          in number,
                            pi_vKey             in varchar2,
                            pi_vKeyValue        in varchar2,
                            po_nStatusCode      out number,
                            po_vStatusComment   out varchar2) 
  is
  
    v_nCount number;
  
  begin
  
    --default status to good
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    v_nCount := 0;
  
    --clear any old values
    delete from tbicds.fx_session_value
     where db_session_id = pi_vSessionID
       and db_session_key = pi_vKey;
    commit;
  
    --check that there is a session matching this session iod and client ip
    select count(*)
      into v_nCount
      from tbicds.fx_session
     where db_session_id = pi_vSessionID
       and client_ip = pi_vSessionClientIP;
  
    --scrap it all, someone has a mismatched ip/sessid combo
    --this will affewctivly log the user out
    if v_nCount < 1 then
    
      delete from tbicds.fx_session where db_session_id = pi_vSessionID;
      commit;
      delete from tbicds.fx_session_value
       where db_session_id = pi_vSessionID;
      commit;
    
      po_nStatusCode    := 1;
      po_vStatusComment := 'SetSessionValue: failed!';
      return;
    
    end if;
  
    --insert a record into fx_session_value
    insert into tbicds.fx_session_value
      (db_session_id, db_session_key, db_session_value, client_ip)
    values
      (pi_vSessionID, pi_vKey, pi_vKeyValue, pi_vSessionClientIP);
  
    commit;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '161 - An error occurred while setting a session value , Please contact your system administrator.';
  end;

  
  /*change password, this will also logg the user in*/
  procedure ChangePassword(pi_vSessionID       in varchar2,
                           pi_vSessionClientIP in varchar2,
                           pi_nUserID          in number, --not used but passed in to maintain consistant default param list
                           pi_vKey             in varchar2,
                           pi_vUserName        in varchar2,
                           pi_vOldPassword     in varchar2,
                           pi_vPassword        in varchar2,
                           pi_vCert            in varchar2,
                           pi_vCOldPassword    in varchar2,
                           pi_vCPassword       in varchar2,
                           pi_vCUserName       in varchar2,
                           po_nUserID          out number,
                           po_vDBSessionID     out varchar2,
                           po_nTimeout         out number,
                           po_nStatusCode      out number,
                           po_vStatusComment   out varchar2)
  
   is
  
    v_nFXUserID            number := 0;
    v_nUserCount           number := 0;
    v_dDatePasswordChanged date;
    v_vPREV_PWD1           varchar2(2000);
    v_vPREV_PWD2           varchar2(2000);
    v_vPREV_PWD3           varchar2(2000);
    v_vPREV_PWD4           varchar2(2000);
    v_vPREV_PWD5           varchar2(2000);
    v_vPREV_PWD6           varchar2(2000);
    v_vPREV_PWD7           varchar2(2000);
    v_vPREV_PWD8           varchar2(2000);
    v_vPREV_PWD9           varchar2(2000);
    v_vPREV_PWD10          varchar2(2000);
  
    --suat user personal information
    v_vUsrPDataName  varchar2(2000);
    v_vUsrPDataName1 varchar2(2000);
    v_vUsrPDataName2 varchar2(2000);
  
    v_vUsrPDataTitle    varchar2(2000);
    v_vUsrPDataUnit     varchar2(2000);
    v_vUsrPDataSquadron varchar2(2000);
    v_vUsrPDataPhone    varchar2(2000);
  
    --patient personal information
    v_vPatPDataFirstName  varchar2(2000);
    v_vPatPDataLastName   varchar2(2000);
    v_vPatPDataSSN        varchar2(2000);
    v_vPatPDataCity       varchar2(2000);
    v_vPatPDataPostalCode varchar2(2000);
    v_vPatPDataHomePhone  varchar2(2000);
    v_vPatPDataWorkPhone  varchar2(2000);
    v_dtPatPDataDOB       date;
    v_vPWDCheck           varchar2(4000);
  
    v_vCUserName1    varchar2(2000);
    v_vCUsername2    varchar2(2000);
    v_vUNCheck       varchar2(2000);
    v_nResetPassword number;
  begin
  
    --default status to good
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
    po_nUserID        := 0;
    v_nFXUserID       := 0;
    v_vPWDCheck       := '';
    v_nResetPassword  := 0;
  
    -- make sure the user only updates his password
    begin
      select count(t.fx_user_id)
        into v_nUserCount
        from tbicds.fx_user t
       where t.fx_user_id = pi_nUserID
         and t.user_name = pi_vUserName;
    exception
      when others then
        po_nStatusCode    := 1;
        po_vStatusComment := '1. Invalid User Name/Password';
        return;
    end;
  
    -- if no records were found the user was trying to update someone else
    if v_nUserCount = 0 and pi_nUserID > 0 then
      po_nStatusCode    := 1;
      po_vStatusComment := 'You may only change your password.';
      return;
    end if;
  
    --check for vaild current password
    begin
      select nvl(t.fx_user_id, 0),
             date_password_changed,
             PREV_PWD1,
             PREV_PWD2,
             PREV_PWD3,
             PREV_PWD4,
             PREV_PWD5,
             PREV_PWD6,
             PREV_PWD7,
             PREV_PWD8,
             PREV_PWD9,
             PREV_PWD10,
             nvl(RESET_PASSWORD, 0)
        into v_nFXUserID,
             v_dDatePasswordChanged,
             v_vPREV_PWD1,
             v_vPREV_PWD2,
             v_vPREV_PWD3,
             v_vPREV_PWD4,
             v_vPREV_PWD5,
             v_vPREV_PWD6,
             v_vPREV_PWD7,
             v_vPREV_PWD8,
             v_vPREV_PWD9,
             v_vPREV_PWD10,
             v_nResetPassword
        from tbicds.fx_user t
       where upper(t.user_name) = upper(pi_vUserName)
         and t.password = pi_vOldPassword;
    
    exception
      when others then
        po_nStatusCode    := 1;
        po_vStatusComment := '2. Invalid User Name/Password';
        return;
    end;
  
    if v_nFXUserID < 1 then
      po_nStatusCode    := 1;
      po_vStatusComment := '3. Invalid User Name/Password';
      return;
    end if;
  
    --validate the password against rules
  
    ValidatePassword(pi_vKey,
                     v_nFXUserID,
                     pi_vUserName,
                     pi_vOldPassword,
                     pi_vPassword,
                     pi_vCOldPassword,
                     pi_vCPassword,
                     pi_vCUserName,
                     v_nResetPassword,
                     po_nStatusCode,
                     po_vStatusComment);
    if po_nStatusCode != 0 then
      return;
    end if;
  
    --save the current password as a past password
    update tbicds.fx_user
       set prev_pwd1  = pi_vOldPassword,
           prev_pwd2  = v_vPREV_PWD1,
           prev_pwd3  = v_vPREV_PWD2,
           prev_pwd4  = v_vPREV_PWD3,
           prev_pwd5  = v_vPREV_PWD4,
           prev_pwd6  = v_vPREV_PWD5,
           prev_pwd7  = v_vPREV_PWD6,
           prev_pwd8  = v_vPREV_PWD7,
           prev_pwd9  = v_vPREV_PWD8,
           prev_pwd10 = v_vPREV_PWD9
     where lower(user_name) = lower(pi_vUserName)
       and password = pi_vOldPassword;
    commit;
  
    --update the users password to the new one
    update tbicds.fx_user
       set password              = pi_vPassword,
           is_locked             = 0,
           is_inactive           = 0,
           date_modified         = sysdate,
           updated_by            = pi_nUserID,
           reset_password        = 0,
           date_password_changed = sysdate,
           login_attempts        = 0
     where lower(user_name) = lower(pi_vUserName)
       and password = pi_vOldPassword;
    commit;
  
    --now login with this new password
    Login(pi_vSessionID,
          pi_vSessionClientIP,
          pi_nUserID,
          pi_vUserName,
          pi_vPassword,
          pi_vCert,
          po_nUserID,
          po_vDBSessionID,
          po_nTimeout,
          po_nStatusCode,
          po_vStatusComment);
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '163 - An error occurred while changing password, Please contact your system administrator.';
  end;

  /*used for signing notes etc.*/
  procedure Sign(pi_vSessionID       in varchar2,
                 pi_vSessionClientIP in varchar2,
                 pi_nUserID          in number,
                 pi_vUserName        in varchar2,
                 pi_vPassword        in varchar2,
                 po_vProviderID      out varchar2,
                 po_nUserType        out number,
                 po_nStatusCode      out number,
                 po_vStatusComment   out varchar2)
  
   is
  v_vStat varchar2(4000);
  v_nStat number;
  
  begin
  
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
    po_vProviderID    := '';
  
    select u.provider_id, r.user_type
      into po_vProviderID, po_nUserType
      from tbicds.fx_user t, tbicds.app_user u, tbicds.fx_user_rights r
     where t.fx_user_id = u.fx_user_id
       and r.fx_user_id = u.fx_user_id
       and t.user_name = pi_vUserName
       and t.password = pi_vPassword;
  
    if po_vProviderID = '' then
    
      --add an audit record
    /*  insert into tbicds.fx_audit
        (db_session_id,
         client_ip,
         fx_user_id,
         audit_date,
         audit_name,
         audit_data)
      values
        (pi_vSessionID,
         pi_vSessionClientIP,
         pi_nUserID,
         sysdate,
         'SIGN',
         'FAIL=' || pi_vUserName);*/
         
      begin
      
        SysLogAuditTransaction(pi_vSessionID,
                               pi_vSessionClientIP,
                               pi_nUserID,
                               1,
                               'SIGN_FAILED'
                               || '|' || 'FAILED',
                                v_nStat,
                                v_vStat);
                             
    
      exception
      when others then
        null;
      end;
    end if;
    
  
  exception
    when others then
      po_nStatusCode := 9;
      --po_vStatusComment := '164 - An error occurred while signing, Please contact your system administrator.';
    
      po_vStatusComment := 'Invalid user name/password';
    
      --add an audit record
      /*insert into tbicds.fx_audit
        (db_session_id,
         client_ip,
         fx_user_id,
         audit_date,
         audit_name,
         audit_data)
      values
        (pi_vSessionID,
         pi_vSessionClientIP,
         pi_nUserID,
         sysdate,
         'SIGN',
         'FAIL=' || pi_vUserName);*/
      
        begin
      
        SysLogAuditTransaction(pi_vSessionID,
                               pi_vSessionClientIP,
                               pi_nUserID,
                               1,
                               'SIGN_FAILED'
                               || '|' || 'FAILED',
                                v_nStat,
                                v_vStat);
                             
    
      exception
      when others then
        null;
      end;
    
  end;

  /*login this is the version that requires a username and password
  after a successful login the persons CAC will be associated with
  this account*/
  procedure Login(pi_vSessionID       in varchar2,
                  pi_vSessionClientIP in varchar2,
                  pi_nUserID          in number, --not used but passed in to maintain consistant default param list
                  pi_vUserName        in varchar2,
                  pi_vPassword        in varchar2,
                  pi_vCert            in varchar2,
                  po_nUserID          out number,
                  po_vDBSessionID     out varchar2,
                  po_nTimeout         out number,
                  po_nStatusCode      out number,
                  po_vStatusComment   out varchar2) 
  is
  
    v_nFXUserID        number;
    v_nFXUserChangePWD number;
    v_nFXUserInactive  number;
    v_nFXUserLocked    number;
  
    v_vDBSessionID    varchar2(4000);
    v_nCurrAttempts   number;
    v_nCount          number;
    v_nCurrIPAttempts number;
    v_nIPIsLocked     number;
    v_dtIPlocked      date;
    v_nIPLockPeriod   number;
    v_dtIPCheck       date;
    v_dtExpires       date;
    v_dtLastLogin     date;
  
    v_nMAX_ACCOUT_LOGIN_ATTEMPTS number;
    v_nMAX_IP_LOGIN_ATTEMPTS     number;
    v_nMAX_IP_LOGIN_TIMEOUT      number;
    v_nStatusCode                number;
    v_vStatus                    varchar2(4000);
    v_vContent                   varchar2(4000);
    v_vAccountCert               varchar2(4000);
  
    v_vstat                      varchar2(4000);
    v_nstat                      number;
    
  begin
  
    --default status to good
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    po_nUserID     := 0;
    v_vDBSessionID := '';
  
    v_nFXUserID        := 0;
    v_nFXUserChangePWD := 0;
    v_nFXUserInactive  := 0;
    v_nFXUserLocked    := 0;
  
    v_vDBSessionID    := '';
    v_nCurrAttempts   := 0;
    v_nCount          := 0;
    v_nCurrIPAttempts := 0;
    v_nIPIsLocked     := 0;
    v_nIPLockPeriod   := 0;
    v_dtIPCheck       := sysdate;
    v_vAccountCert    := '';
  
    --
    v_nMAX_ACCOUT_LOGIN_ATTEMPTS := 3;
    v_nMAX_IP_LOGIN_ATTEMPTS     := 10;
    v_nMAX_IP_LOGIN_TIMEOUT      := 20;
  
    --
    --check to see if we are IP locked
    --
    begin
    
      --is the ip locked
      select nvl(t.is_locked, 0)
        into v_nIPIsLocked
        from tbicds.fx_iplogin_attempts t
       where t.client_ip = pi_vSessionClientIP;
    
      --time period in mins to lock
      select nvl(t.lock_period, v_nMAX_IP_LOGIN_TIMEOUT)
        into v_nIPLockPeriod
        from tbicds.fx_iplogin_attempts t
       where t.client_ip = pi_vSessionClientIP;
    
      --date locked
      select nvl(t.date_last_attempt, sysdate)
        into v_dtIPlocked
        from tbicds.fx_iplogin_attempts t
       where t.client_ip = pi_vSessionClientIP;
      if v_nIPIsLocked > 0 then
      
        --if the current date/time > (lock time + lock period)
        v_dtIPCheck := v_dtIPlocked + v_nIPLockPeriod / 1440;
        if sysdate > v_dtIPCheck then
        
          --if we are no longer ip locked clear the
          --fx_iplogin_attempts entry
          delete from tbicds.fx_iplogin_attempts
           where client_ip = pi_vSessionClientIP;
          commit;
        
        else
        
          po_nUserID        := '0';
          po_nStatusCode    := 6;
          po_vStatusComment := 'IP Address Locked!';
          return;
        
        end if;
      
      end if;
    
    exception
      when others then
        v_nIPIsLocked := 0;
    end;
  
    --login
    begin
    
      --get the user id and status info about the account
      select nvl(t.fx_user_id, 0),
             nvl(t.reset_password, 0),
             nvl(t.is_inactive, 0),
             nvl(t.is_locked, 0),
             t.date_password_changed,
             t.date_last_login
        into v_nFXUserID,
             v_nFXUserChangePWD,
             v_nFXUserInactive,
             v_nFXUserLocked,
             v_dtExpires,
             v_dtLastLogin
       
        from tbicds.fx_user t
       where upper(t.user_name) = upper(pi_vUserName)
         and t.password = pi_vPassword;
    
    exception
      when others then
      
        v_nFXUserID := 0;
      
    end;
  
    if v_nFXUserID > 0 then
    
      --check to see if this user is locked...
      if v_nFXUserLocked > 0 then
        po_nUserID        := '0';
        po_nStatusCode    := 2;
        po_vStatusComment := 'Your account is locked!';
        return;
      end if;
    
      --check to see if the user is inactive
      if v_nFXUserInactive > 0 then
        po_nUserID        := '0';
        po_nStatusCode    := 3;
        po_vStatusComment := 'Your account is inactive!';
        return;
      end if;
    
      --check the date_password_changed value and compare it
      --to the rules for pwd expiration...
      if v_dtExpires < trunc(sysdate) - 60 then
      
        po_nUserID        := '0';
        po_nStatusCode    := 9;
        po_vStatusComment := 'Your account has expired! Please contact your System Administrator to reset your password. ';
        return;
      
      end if;
    
      --make sure the user has logged in in the last 35 days otherwise they are expired
      if v_dtLastLogin < trunc(sysdate) - 35 and
         v_dtExpires < trunc(sysdate) - 35 then
      
        po_nUserID        := '0';
        po_nStatusCode    := 9;
        po_vStatusComment := 'Your account has expired due to inactivity! Please contact your System Administrator to reset your password. ';
        return;
      
      end if;
    
      --make sure there is not already a cert attached to this account
      if v_vAccountCert is not null then
        if v_vAccountCert != pi_vCert then
        
          po_nUserID        := '0';
          po_nStatusCode    := 9;
          po_vStatusComment := 'There is already a certificate associated with this account! Please contact your System Administrator to reset your password. ';
          return;
        
        end if;
      end if;
    
      --update the cert so that we autologin next time
     
      --reset the fx_user status info
      update tbicds.fx_user
         set is_locked       = 0,
             is_inactive     = 0,
             date_last_login = sysdate,
             flogin_attempts = login_attempts,
             login_attempts  = 0,
             LAST_LOGIN_IP   = pi_vSessionClientIP
       where fx_user_id = v_nFXUserID;
      commit;
    
      --reset the ip login attempts
      delete from tbicds.fx_iplogin_attempts
       where client_ip = pi_vSessionClientIP;
      commit;
    
      --set the timeout
      po_nTimeOut := 15;
      begin
      
        select nvl(t.session_timeout, 15)
          into po_nTimeout
          from tbicds.fx_user t
         where fx_user_id = v_nFXUserID;
      exception
        when others then
        
          po_nTimeOut := 0;
        
      end;
    
      --check to see if the user needs to change pwd
      if v_nFXUserChangePWD > 0 then
        po_nStatusCode    := 4;
        po_vStatusComment := 'Please Change Password!';
        return;
      end if;
    
      --get a new db session id:
      v_vDBSessionID := '';
      select t.SID || DBMS_CRYPTO.RandomBytes(3) || t.SERIAL# || t.USER# ||
             DBMS_CRYPTO.RandomBytes(3) || userenv('sessionid')
        into v_vDBSessionID
        from sys.v_$session t
       where t.audsid = userenv('sessionid');
    
      --clean up any old sessions, users can only login once!!!
      delete from tbicds.fx_session where fx_user_id = v_nFXUserID;
      commit;
      delete from tbicds.fx_session where db_session_id = v_vDBSessionID;
      commit;
      delete from tbicds.fx_session_value
       where db_session_id = v_vDBSessionID;
      commit;
    
      --insert a record into fx_session
      insert into tbicds.fx_session
        (fx_user_id,
         web_session_id,
         db_session_id,
         date_created,
         date_last_action,
         expired,
         client_ip)
      values
        (v_nFXUserID,
         pi_vSessionID,
         v_vDBSessionID,
         sysdate,
         sysdate,
         0,
         pi_vSessionClientIP);
    
      commit;
    
      --add an audit record for the success
    /*  insert into tbicds.fx_audit
        (db_session_id,
         client_ip,
         fx_user_id,
         audit_date,
         audit_name,
         audit_data)
      values
        (v_vDBSessionID,
         pi_vSessionClientIP,
         v_nFXUserID,
         sysdate,
         'LOGIN',
         'SUCCESS');
    
      commit;*/
      begin
      
        SysLogAuditTransaction(v_vDBSessionID,
                               pi_vSessionClientIP,
                               pi_nUserID,
                               0,
                               'LOGIN'
                               || '|' || 'SUCCESS',
                                v_nStat,
                                v_vStat);
                             
    
      exception
      when others then
        null;
      end;
    
      --return the user id
      po_nUserID      := v_nFXUserID;
      po_vDBSessionID := v_vDBSessionID;
    
    else
    
      po_nUserID        := '0';
      po_nStatusCode    := 1;
      po_vStatusComment := 'Invalid user name/password';
    
      --did we fail the login but have a valid user name?
      select count(*)
        into v_nCount
        from tbicds.fx_user t
       where upper(t.user_name) = upper(pi_vUserName);
      if v_nCount > 0 then
      
        --get the number of current attempts
        begin
          select nvl(t.login_attempts, 0)
            into v_nCurrAttempts
            from tbicds.fx_user t
           where upper(t.user_name) = upper(pi_vUserName);
        exception
          when others then
            v_nCurrAttempts := 0;
        end;
      
        --increment number of login attempts
        v_nCurrAttempts := v_nCurrAttempts + 1;
      
        --update the fx_user record
        update tbicds.fx_user
           set login_attempts   = v_nCurrAttempts,
               LAST_FLOGIN_IP   = pi_vSessionClientIP,
               LAST_FLOGIN_DATE = sysdate
         where upper(user_name) = upper(pi_vUserName);
      
        --check to see if we need to lock the account
        if v_nCurrAttempts >= v_nMAX_ACCOUT_LOGIN_ATTEMPTS then
        
          update tbicds.fx_user
             set is_locked = 1
           where upper(user_name) = upper(pi_vUserName);
          commit;
        
          po_nUserID        := '0';
          po_nStatusCode    := 7;
          po_vStatusComment := 'Invalid user name/password, Account Locked!';
        
        end if;
      
      else
      
        --attempting to login with a bad user name
        --need to track ip address and lock that ip
        --for a specified period of time
        v_nCount := 0;
        select count(*)
          into v_nCount
          from tbicds.FX_IPLOGIN_ATTEMPTS t
         where client_ip = pi_vSessionClientIP;
        if v_nCount > 0 then
        
          --get the number of attempts from this ip
          begin
            select login_attempts
              into v_nCurrIPAttempts
              from tbicds.FX_IPLOGIN_ATTEMPTS
             where client_ip = pi_vSessionClientIP;
          exception
            when others then
              v_nCurrIPAttempts := 0;
          end;
        
          --increment attempts
          v_nCurrIPAttempts := v_nCurrIPAttempts + 1;
        
          --update last attempt and date
          update tbicds.FX_IPLOGIN_ATTEMPTS
             set date_last_attempt = sysdate,
                 login_attempts    = v_nCurrIPAttempts
           where client_ip = pi_vSessionClientIP;
          commit;
        
          --check and lock if necessary
          if v_nCurrIPAttempts >= v_nMAX_IP_LOGIN_ATTEMPTS then
          
            update tbicds.FX_IPLOGIN_ATTEMPTS
               set is_locked = 1
             where client_ip = pi_vSessionClientIP;
            commit;
          
            po_nUserID        := '0';
            po_nStatusCode    := 3;
            po_vStatusComment := 'Invalid user name/password, IP Address Locked!';
          
            --email the SA and let them know...
            v_vContent := '';
            v_vContent := 'IP Address ' || pi_vSessionClientIP || ' ';
            v_vContent := v_vContent ||
                          'attempted to login 10 times with invalid user account information! ';
          end if;
        
        else
        
          --first bad ip attempt
          insert into tbicds.FX_IPLOGIN_ATTEMPTS
            (client_ip,
             login_attempts,
             date_last_attempt,
             is_locked,
             lock_period)
          values
            (pi_vSessionClientIP, 1, sysdate, 0, v_nMAX_IP_LOGIN_TIMEOUT);
          commit;
        
        end if;
      
      end if;
    
      --add an audit record for the fail
      /*insert into tbicds.fx_audit
        (db_session_id,
         client_ip,
         fx_user_id,
         audit_date,
         audit_name,
         audit_data)
      values
        (v_vDBSessionID,
         pi_vSessionClientIP,
         -1,
         sysdate,
         'LOGIN',
         'FAIL=' || pi_vUserName);*/
         
      begin
      
        SysLogAuditTransaction(v_vDBSessionID,
                               pi_vSessionClientIP,
                               pi_nUserID,
                               1,
                               'LOGIN_FAILED'
                               || '|' || 'FAILED',
                                v_nStat,
                                v_vStat);
                             
    
      exception
      when others then
        null;
      end;
    
    end if;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '165 - An error occurred while logging in, Please contact your system administrator.';
    
      --add an audit record
      /*insert into tbicds.fx_audit
        (db_session_id,
         client_ip,
         fx_user_id,
         audit_date,
         audit_name,
         audit_data)
      values
        (v_vDBSessionID,
         pi_vSessionClientIP,
         -1,
         sysdate,
         'LOGIN',
         'FAIL=' || pi_vUserName);*/
         
       begin
      
        SysLogAuditTransaction(v_vDBSessionID,
                               pi_vSessionClientIP,
                               pi_nUserID,
                               1,
                               'LOGIN_FAILED'
                               || '|' || 'FAILED',
                                v_nStat,
                                v_vStat);
                             
    
      exception
      when others then
        null;
      end;
      
  end;

  /*audit page acceess*/
  procedure AuditPageAccess(pi_vSessionID       in varchar2,
                            pi_vSessionClientIP in varchar2,
                            pi_nUserID          in number,
                            pi_vPageName        in varchar2,
                            po_nStatusCode      out number,
                            po_vStatusComment   out varchar2)
  
   is
    v_nCount        number;
    v_nPageUserType number;
    v_nPageRights   number;
    v_nUserType     number;
    v_nUserRights   number;
    v_nSessionCount number;
    v_nPageActive   number;
    v_nPageLoggedIn number;
    v_nStat         number;
    v_vStat         varchar2(4000);
  
  begin
    v_nSessionCount := 0;
    v_nPageUserType := 0;
    v_nPageRights   := 0;
    v_nUserType     := 0;
    v_nUserRights   := 0;
    v_nPageLoggedIn := 0;
  
    --default status to good
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    v_nCount := 0;
  
    select nvl(t.user_rights, 0),
           nvl(t.user_type, 0),
           nvl(t.active, 1),
           nvl(t.is_logged_in, 0)
      into v_nPageRights, v_nPageUserType, v_nPageActive, v_nPageLoggedIn
      from tbicds.fx_page_access t
     where t.page_name = pi_vPageName;
  
    /*insert into tbicds.fx_audit
      (db_session_id,
       client_ip,
       fx_user_id,
       audit_date,
       audit_name,
       audit_data)
    values
      (pi_vSessionID,
       pi_vSessionClientIP,
       pi_nUserID,
       sysdate,
       'PAGE_ACCESS: ' || pi_vPageName,
       '');*/
       
      begin
      
        SysLogAuditTransaction(pi_vSessionID,
                               pi_vSessionClientIP,
                               pi_nUserID,
                               0,
                               'PAGE_ACCESS: ' 
                               || '|' || pi_vSessionClientIP
                               || '|' || pi_nUserID
                               || '|' || pi_vPageName
                               || '|' || 'SUCCESS',
                                v_nStat,
                                v_vStat);
                             
    
      exception
      when others then
        null;
      end;
  
    commit;
  
    -- There are some pages disabled in this version
    if v_nPageActive = 0 then
    
      --if we get here then we need to fail
      po_nStatusCode    := 1;
      po_vStatusComment := 'You are not authorized to view this page!';
    
      /*insert into tbicds.fx_audit
        (db_session_id,
         client_ip,
         fx_user_id,
         audit_date,
         audit_name,
         audit_data)
      values
        (pi_vSessionID,
         pi_vSessionClientIP,
         pi_nUserID,
         sysdate,
         'FAILED_PAGE_ACCESS: ' || pi_vPageName,
         '');*/
         
      begin
      
        SysLogAuditTransaction(pi_vSessionID,
                               pi_vSessionClientIP,
                               pi_nUserID,
                               1,
                               'FAILED_PAGE_ACCESS: ' 
                               || '|' || pi_vSessionClientIP
                               || '|' || pi_nUserID
                               || '|' || pi_vPageName
                               || '|' || 'FAILED',
                                v_nStat,
                                v_vStat);
                             
    
      exception
      when others then
        null;
      end;
    
      commit;
      return;
    
    end if;
  
    --allow access to everyone -not logged in
    if (v_nPageLoggedIn = 0) then
      --we are done
      return;
    end if;
  
    --allow access to everyone logged in
    if (v_nPageRights = 0) and (v_nPageUserType = 0) then
      --we are done
      return;
    end if;
  
    --
    --
    --all other pages from this point on require you to be logged in
    --and have some kind of permission
    --
    --
  
    --check that we have a valid user id
    if pi_nUserID < 1 then
    
      --if we get here then we need to fail
      po_nStatusCode    := 1;
      po_vStatusComment := 'You are not authorized to view this page!';
    
      /*insert into tbicds.fx_audit
        (db_session_id,
         client_ip,
         fx_user_id,
         audit_date,
         audit_name,
         audit_data)
      values
        (pi_vSessionID,
         pi_vSessionClientIP,
         pi_nUserID,
         sysdate,
         'FAILED_PAGE_ACCESS: ' || pi_vPageName,
         '');*/
         
        begin
      
        SysLogAuditTransaction(pi_vSessionID,
                               pi_vSessionClientIP,
                               pi_nUserID,
                               1,
                               'FAILED_PAGE_ACCESS: ' 
                               || '|' || pi_vSessionClientIP
                               || '|' || pi_nUserID
                               || '|' || pi_vPageName
                               || '|' || 'FAILED',
                                v_nStat,
                                v_vStat);
                             
    
      exception
      when others then
        null;
      end;
  
      commit;
      return;
    
    end if;
  
    --check that we have a valid session record
    select count(*)
      into v_nSessionCount
      from tbicds.fx_session t
     where t.fx_user_id = pi_nUserID
       and t.expired = 0;
    if v_nSessionCount != 1 then
    
      --if we get here then we need to fail
      po_nStatusCode    := 1;
      po_vStatusComment := 'You are not authorized to view this page!';
    
      /*insert into tbicds.fx_audit
        (db_session_id,
         client_ip,
         fx_user_id,
         audit_date,
         audit_name,
         audit_data)
      values
        (pi_vSessionID,
         pi_vSessionClientIP,
         pi_nUserID,
         sysdate,
         'FAILED_PAGE_ACCESS: ' || pi_vPageName,
         '');*/
       
      begin
      
         SysLogAuditTransaction(pi_vSessionID,
                               pi_vSessionClientIP,
                               pi_nUserID,
                               1,
                               'FAILED_PAGE_ACCESS: ' 
                               || '|' || pi_vSessionClientIP
                               || '|' || pi_nUserID
                               || '|' || pi_vPageName
                               || '|' || 'FAILED',
                                v_nStat,
                                v_vStat);
                             
    
      exception
      when others then
        null;
      end;
      
      commit;
      return;
    
    end if;
  
    --otherwise we must check
    select t.user_type, t.user_rights
      into v_nUserType, v_nUserRights
      from tbicds.fx_user_rights t
     where t.fx_user_id = pi_nUserID;
  
    if (((bitand(v_nUserRights, v_nPageRights) > 0) or v_nPageRights = 0) and
       ((bitand(v_nUserType, v_nPageUserType) > 0) or v_nPageUserType = 0))
    
     then
      return;
    end if;
  
    --if we get here then we need to fail
    po_nStatusCode    := 1;
    po_vStatusComment := 'You are not authorized to view this page!';
  
    /*insert into tbicds.fx_audit
      (db_session_id,
       client_ip,
       fx_user_id,
       audit_date,
       audit_name,
       audit_data)
    values
      (pi_vSessionID,
       pi_vSessionClientIP,
       pi_nUserID,
       sysdate,
       'FAILED_PAGE_ACCESS: ' || pi_vPageName,
       '');*/
       
       begin
      
        SysLogAuditTransaction(pi_vSessionID,
                               pi_vSessionClientIP,
                               pi_nUserID,
                               1,
                               'FAILED_PAGE_ACCESS: ' 
                               || '|' || pi_vSessionClientIP
                               || '|' || pi_nUserID
                               || '|' || pi_vPageName
                               || '|' || 'FAILED',
                                v_nStat,
                                v_vStat);
                             
    
      exception
      when others then
        null;
      end;
  
    commit;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '166 - An error occurred while auditing page access , Please contact your system administrator.';
    
      /*insert into tbicds.fx_audit
        (db_session_id,
         client_ip,
         fx_user_id,
         audit_date,
         audit_name,
         audit_data)
      values
        (pi_vSessionID,
         pi_vSessionClientIP,
         pi_nUserID,
         sysdate,
         'FAILED_PAGE_ACCESS: ' || pi_vPageName,
         '');*/
      
        begin
      
         SysLogAuditTransaction(pi_vSessionID,
                               pi_vSessionClientIP,
                               pi_nUserID,
                               1,
                               'FAILED_PAGE_ACCESS: ' 
                               || '|' || pi_vSessionClientIP
                               || '|' || pi_nUserID
                               || '|' || pi_vPageName
                               || '|' || 'FAILED',
                                v_nStat,
                                v_vStat);
                             
    
      exception
      when others then
        null;
      end;
    
      commit;
  end;

  /*audits a transaction by encrypting the clob passed in 
  and storing the params and clob in the fx_audit table*/
  procedure AuditTransaction(pi_vSessionID       in varchar2,
                             pi_vSessionClientIP in varchar2,
                             pi_nUserID          in number,
                             pi_vKey             in varchar2,
                             pi_vSPName          in varchar2,
                             pi_clAuditXML       in clob,
                             po_nStatusCode      out number,
                             po_vStatusComment   out varchar2)
  
   is
  
    v_nCount   number := 0;
    v_nCounter number := 1;
  
    --the encrypted audit text
    v_vAudit varchar(32767) := '';
  
    v_vWorking varchar(32767) := '';
  
  begin
  
    --default status to good
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    v_nCount := 0;
  
    --if audit is < 4001 chars then enc the whole thing
    --otherwise enc it in pieces seperated by _ most calls will 
    --be under 4K!!!
    if length(pi_clAuditXML) < 2001 then
    
      v_vAudit := ic_utl_sec.encryptData(substr(pi_clAuditXML, 1, 2000),
                                             pi_vKey,
                                             pi_vSessionClientIP);
    else
    
      --get a working string
      v_vWorking := pi_clAuditXML;
      v_vAudit   := '';
    
      --how many times do we need to chunk it
      v_nCount := length(v_vWorking) / 1000;
      v_nCount := ceil(v_nCount);
    
      --loop and chunk
      for v_nCounter IN 1 .. v_nCount loop
      
        if v_nCounter = v_nCount then
        
          --last one so grab the rest
          v_vAudit := v_vAudit ||
                      ic_utl_sec.encryptData(v_vWorking,
                                                 pi_vKey,
                                                 pi_vSessionClientIP);
          v_vAudit := v_vAudit || '|';
        
        else
        
          --grab 1000 chars  
          v_vAudit := v_vAudit ||
                      ic_utl_sec.encryptData(substr(v_vWorking, 1, 1000),
                                                 pi_vKey,
                                                 pi_vSessionClientIP);
          v_vAudit := v_vAudit || '|';
        
          --adjust working to next 4k group
          v_vWorking := substr(v_vWorking, 1001);
        
        end if;
      end loop;
    
    end if;
  
    --insert the record into the audit table
    insert into tbicds.fx_audit
      (db_session_id,
       client_ip,
       fx_user_id,
       audit_date,
       audit_name,
       audit_data)
    values
      (pi_vSessionID,
       pi_vSessionClientIP,
       pi_nUserID,
       sysdate,
       pi_vSPName,
       v_vAudit);
  
    commit;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '167 - An error occurred while auditing a transaction, Please contact your system administrator.';
  end;

  
  /*Log Off, clears session from the db*/
  procedure LogOff(pi_vSessionID       in varchar2,
                   pi_vSessionClientIP in varchar2,
                   pi_nUserID          in number,
                   po_nStatusCode      out number,
                   po_vStatusComment   out varchar2)
  
   is
    v_nCount number;
  
  begin
  
    --default status to good
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    v_nCount := 0;
  
    --clean up any old sessions, users can only login once!!!
    delete from tbicds.fx_session where fx_user_id = pi_nUserID;
    commit;
    delete from tbicds.fx_session where db_session_id = pi_vSessionID;
    commit;
    delete from tbicds.fx_session_value
     where db_session_id = pi_vSessionID;
    commit;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '168 - An error occurred while logging off, Please contact your system administrator.';
  end;

  /*gets an fx_user record given the encrypted uid*/
  procedure GetFXUserRS(pi_vSessionID       in varchar2,
                        pi_vSessionClientIP in varchar2,
                        pi_nUserID          in number,
                        pi_vEncUID          in varchar2,
                        po_nStatusCode      out number,
                        po_vStatusComment   out varchar2,
                        rs                  out RetRefCursor) 
  is
  
  begin
  
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    --open recordset
    open rs for
      select t.*,
             --less than or equal to zero means we are expired!
             floor(60 - (trunc(sysdate) - t.date_password_changed)) as days_till_expiration
        from tbicds.fx_user t
       where user_name = pi_vEncUID;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '169 - An error occurred while retrieving a user record, Please contact your system administrator.';
  end;

  /*insert a record into the fx_user table
  and update the fx_user_id in the app_user table*/
  procedure InsertFXUser(pi_vSessionID       in varchar2,
                         pi_vSessionClientIP in varchar2,
                         pi_nUserID          in number,
                         pi_vKey             in varchar2,
                         pi_vProviderID      in varchar2,
                         pi_vUserName        in varchar2,
                         pi_vPassword        in varchar2,
                         pi_nAccountLocked   in number,
                         pi_nAccountInactive in number,
                         
                         pi_vCOldPassword in varchar2,
                         pi_vCPassword    in varchar2,
                         pi_vCUserName    in varchar2,
                         
                         po_nFXUserID      out number,
                         po_nStatusCode    out number,
                         po_vStatusComment out varchar2)
  
   is
    v_nCount    number;
    v_nFXUserID number;
    dtSys       date;
  
  begin
  
    --default status to good
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    v_nCount    := 0;
    v_nFXUserID := 0;
    dtSys       := sysdate;
  
    --get a new user id from the sequence
    --
    select tbicds.seqFXUserID.Nextval into v_nFXUserID from dual;
  
    insert into tbicds.fx_user
      (fx_user_id,
       user_name,
       is_locked,
       is_inactive,
       date_created,
       date_modified,
       updated_by,
       reset_password,
       date_password_changed,
       date_last_login,
       session_timeout,
       login_attempts,
       password)
    values
      (v_nFXUserID, --fx_user_id,
       pi_vUserName, --user_name,
       pi_nAccountLocked, --is_locked,
       pi_nAccountInactive, --is_inactive,
       dtSys, --date_created,
       dtSys, --date_modified,
       pi_nUserID, --updated_by,
       1, --ALWAYS MUST reset_password,
       dtSys, --date_password_changed,
       dtSys, --date_last_login,
       15, --session_timeout,
       0, --login_attempts,
       pi_vPassword --password
       );
  
    commit;
  
    --update the app_user table fx id
    update tbicds.app_user
       set fx_user_id = v_nFXUserID
     where provider_id = pi_vProviderID;
    commit;
  
    --create user record in fx_user_rights
    insert into tbicds.fx_user_rights
      (fx_user_id, user_rights, read_only)
    values
      (v_nFXUserID, 0, 0);
    commit;
  
    --have to insert the record before validating
    ValidatePassword(pi_vKey,
                     v_nFXUserID,
                     pi_vUserName,
                     'NA',
                     pi_vPassword,
                     'NA',
                     pi_vCPassword,
                     pi_vCUserName,
                     1,
                     po_nStatusCode,
                     po_vStatusComment);
  
    if po_nStatusCode != 0 then
    
      delete from tbicds.fx_user where fx_user_id = v_nFXUserID;
      commit;
    
      po_nFXUserID := 0;
    
    end if;
  
    po_nFXUserID := v_nFXUserID;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '170 - An error occurred while inserting a user, Please contact your system administrator.';
  end;

  /*update an fx_user record*/
  procedure UpdateFXUser(pi_vSessionID       in varchar2,
                         pi_vSessionClientIP in varchar2,
                         pi_nUserID          in number,
                         pi_nFXUserID        in number,
                         pi_vProviderID      in varchar2,
                         pi_vUserName        in varchar2,
                         pi_vPassword        in varchar2,
                         pi_nAccountLocked   in number,
                         pi_nAccountInactive in number,
                         po_nStatusCode      out number,
                         po_vStatusComment   out varchar2)
  
   is
    v_nCount  number;
    dtSys     date;
    v_currpwd varchar2(4000);
  
  begin
  
    --default status to good
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    v_nCount  := 0;
    dtSys     := sysdate;
    v_currpwd := '';
  
    --update date_password_changed if the user changed the pwd
    select password
      into v_currpwd
      from tbicds.fx_user
     where fx_user_id = pi_nFXUserID;
    if v_currpwd != pi_vPassword then
    
      update tbicds.fx_user
         set date_password_changed = dtSys
       where fx_user_id = pi_nFXUserID;
      commit;
    
    end if;
  
    update tbicds.fx_user
       set is_locked      = pi_nAccountLocked,
           is_inactive    = pi_nAccountInactive,
           date_modified  = dtSys,
           updated_by     = pi_nUserID,
           reset_password = 1, --always MUST CHANGE PASSWORD
           password       = pi_vPassword
     where fx_user_id = pi_nFXUserID;
    commit;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '171 - An error occurred while updating a user, Please contact your system administrator.';
  end;

  /*update fx_user_rights*/
  procedure UpdateFXUserRights(pi_vSessionID       in varchar2,
                               pi_vSessionClientIP in varchar2,
                               pi_nUserID          in number,
                               pi_nFxUserID        in number,
                               pi_nUserType        in number,
                               pi_nUserRights      in number,
                               pi_nUserReadOnly    in number,
                               po_nStatusCode      out number,
                               po_vStatusComment   out varchar2) 
 is
  
  begin
  
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    update tbicds.fx_user_rights
       set user_rights = pi_nUserRights,
           read_only   = pi_nUserReadOnly,
           user_type   = pi_nUserType
     where fx_user_id = pi_nFxUserID;
    commit;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_FX_SEC.UpdateSuatUserRights(): ';
  end;

  /*update an fx_user record pwd*/
  procedure UpdateFXUserPWD(pi_vSessionID       in varchar2,
                            pi_vSessionClientIP in varchar2,
                            pi_nUserID          in number,
                            pi_vKey             in varchar2,
                            pi_nFXUserID        in number,
                            pi_vUserName        in varchar2,
                            pi_vPassword        in varchar2,
                            pi_nAccountLocked   in number,
                            pi_nAccountInactive in number,
                            
                            pi_vCPassword in varchar2,
                            pi_vCUserName in varchar2,
                            
                            po_nStatusCode    out number,
                            po_vStatusComment out varchar2)
  
   is
    v_nCount  number;
    dtSys     date;
    v_currpwd varchar2(4000);
  
  begin
  
    --default status to good
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    v_nCount  := 0;
    dtSys     := sysdate;
    v_currpwd := '';
  
    ValidatePassword(pi_vKey,
                     pi_nFXUserID,
                     pi_vUserName,
                     'NA',
                     pi_vPassword,
                     'NA',
                     pi_vCPassword,
                     pi_vCUserName,
                     1,
                     po_nStatusCode,
                     po_vStatusComment);
  
    if po_nStatusCode != 0 then
      return;
    end if;
  
    --update date_password_changed if the user changed the pwd
    select password
      into v_currpwd
      from tbicds.fx_user
     where fx_user_id = pi_nFXUserID;
    if v_currpwd != pi_vPassword then
    
      update tbicds.fx_user
         set date_password_changed = dtSys
       where fx_user_id = pi_nFXUserID;
      commit;
    
    end if;
  
    update tbicds.fx_user
       set is_locked      = pi_nAccountLocked,
           is_inactive    = pi_nAccountInactive,
           date_modified  = dtSys,
           updated_by     = pi_nUserID,
           reset_password = 1, --always MUST CHANGE PASSWORD
           password       = pi_vPassword
     where fx_user_id = pi_nFXUserID;
    commit;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '172 - An error occurred while resetting a password, Please contact your system administrator.';
  end;

  /*update an fx_user options*/
  procedure UpdateFXUserOptions(pi_vSessionID       in varchar2,
                                pi_vSessionClientIP in varchar2,
                                pi_nUserID          in number,
                                pi_nFXUserID        in number,
                                pi_nAccountLocked   in number,
                                pi_nAccountInactive in number,
                                po_nStatusCode      out number,
                                po_vStatusComment   out varchar2)
  
   is
    dtSys date;
  
  begin
  
    --default status to good
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    dtSys := sysdate;
  
    update tbicds.fx_user
       set is_locked     = pi_nAccountLocked,
           is_inactive   = pi_nAccountInactive,
           date_modified = dtSys,
           updated_by    = pi_nUserID
     where fx_user_id = pi_nFXUserID;
    commit;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '173 - An error occurred while updating a user account, Please contact your system administrator.';
  end;

  /*gets an Username and Password given the providerid*/
  procedure GetFXUsernamePasswordRS(pi_vSessionID       in varchar2,
                                    pi_vSessionClientIP in varchar2,
                                    pi_nUserID          in number,
                                    pi_vProviderID      in varchar2,
                                    po_nStatusCode      out number,
                                    po_vStatusComment   out varchar2,
                                    rs                  out RetRefCursor) 
  is
  
  begin
  
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    --open recordset
    open rs for
      select * from tbicds.fx_user t where fx_user_id = pi_nUserID;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '174 - An error occurred while retrieving a user account, Please contact your system administrator.';
  end;

  /* gets an FXUserID given the providerid */
  procedure GetFXUserIdRS(pi_vSessionID       in varchar2,
                          pi_vSessionClientIP in varchar2,
                          pi_nUserID          in number,
                          pi_vProviderID      in varchar2,
                          po_nStatusCode      out number,
                          po_vStatusComment   out varchar2,
                          rs                  out RetRefCursor)
  
   is
  
  begin
  
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    --open recordset
    open rs for
      select fx_user_id
        from tbicds.app_user t
       where provider_id = pi_vProviderID;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '175 - An error occurred while retrieving a user ID, Please contact your system administrator.';
  end;

  /*gets a FXUserID given the providerid*/
  procedure CheckFXUserRecRS(pi_vSessionID       in varchar2,
                             pi_vSessionClientIP in varchar2,
                             pi_nUserID          in number,
                             pi_vProviderID      in varchar2,
                             po_nStatusCode      out number,
                             po_vStatusComment   out varchar2,
                             rs                  out RetRefCursor) 
   is
  
  begin
  
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    --open recordset
    open rs for
      select count(f.fx_user_id) as FXUserCount
        from tbicds.fx_user f
       where f.fx_user_id =
             (select s.fx_user_id
                from tbicds.app_user s
               where s.provider_id = pi_vProviderID);
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '176 - An error occurred while checking a user record, Please contact your system administrator.';
  end;

  /* gets user's selected security questions */
  procedure GetUserQuestions(pi_vSessionID       in varchar2,
                               pi_vSessionClientIP in varchar2,
                               pi_nUserID          in number,
                               --
                               pi_vUsername in varchar2,
                               --                                 
                               po_nStatusCode    out number,
                               po_vStatusComment out varchar2,
                               rs                out RetRefCursor) 
                               
    is
    
      v_nFXUserID number := 0;
      v_nCount    number := 0;  
      
      v_nCurrIPAttempts number;
      v_nIPIsLocked     number;
      v_dtIPlocked      date;
      v_nIPLockPeriod   number;
      v_dtIPCheck       date;
      v_dtExpires       date;
      v_dtLastLogin     date;
      
      v_nMAX_IP_LOGIN_ATTEMPTS     number;
      v_nMAX_IP_LOGIN_TIMEOUT      number;
    
    begin    
         
      v_nCurrIPAttempts := 0;
      v_nIPIsLocked     := 0;
      v_nIPLockPeriod   := 0;
      v_dtIPCheck       := sysdate;
      
      v_nMAX_IP_LOGIN_ATTEMPTS     := 10;
      v_nMAX_IP_LOGIN_TIMEOUT      := 20;
    
      po_nStatusCode    := 0; --0 = success
      po_vStatusComment := '';
    
      --
      --check to see if we are IP locked
      --
      begin
      
        --is the ip locked
        select nvl(t.is_locked, 0)
          into v_nIPIsLocked
          from tbicds.fx_iplogin_attempts t
         where t.client_ip = pi_vSessionClientIP;
      
        --time period in mins to lock
        select nvl(t.lock_period, v_nMAX_IP_LOGIN_TIMEOUT)
          into v_nIPLockPeriod
          from tbicds.fx_iplogin_attempts t
         where t.client_ip = pi_vSessionClientIP;
      
        --date locked
        select nvl(t.date_last_attempt, sysdate)
          into v_dtIPlocked
          from tbicds.fx_iplogin_attempts t
         where t.client_ip = pi_vSessionClientIP;
        if v_nIPIsLocked > 0 then
        
          --if the current date/time > (lock time + lock period)
          v_dtIPCheck := v_dtIPlocked + v_nIPLockPeriod / 1440;
          if sysdate > v_dtIPCheck then
          
            --if we are no longer ip locked clear the
            --fx_iplogin_attempts entry
            delete from tbicds.fx_iplogin_attempts
             where client_ip = pi_vSessionClientIP;
            commit;
          
          else
          
            open rs
             for select (select 0 from dual) as fx_user_id,
                        (select '' from dual) as question_1,
                        (select '' from dual) as question_2,
                        (select 0 from dual) as is_locked,
                        (select 0 from dual) as is_locked
                   from dual;
                 
            po_nStatusCode    := 6;
            po_vStatusComment := 'IP Address Locked!';
            return;
          
          end if;
        
        end if;
      
      exception
        when others then
          v_nIPIsLocked := 0;
      end;
      
      select count(*)
        into v_nCount
        from tbicds.fx_user t
       where t.user_name = pi_vUsername;
    
      if v_nCount > 0 then
        select fx_user_id
          into v_nFXUserID
          from tbicds.fx_user t
         where t.user_name = pi_vUsername;
      else
        --attempting to login with a bad user name
          --need to track ip address and lock that ip
          --for a specified period of time
          v_nCount := 0;
          select count(*)
            into v_nCount
            from tbicds.FX_IPLOGIN_ATTEMPTS t
           where client_ip = pi_vSessionClientIP;
          if v_nCount > 0 then
          
            --get the number of attempts from this ip
            begin
              select login_attempts
                into v_nCurrIPAttempts
                from tbicds.FX_IPLOGIN_ATTEMPTS
               where client_ip = pi_vSessionClientIP;
            exception
              when others then
                v_nCurrIPAttempts := 0;
            end;
          
            --increment attempts
            v_nCurrIPAttempts := v_nCurrIPAttempts + 1;
          
            --update last attempt and date
            update tbicds.FX_IPLOGIN_ATTEMPTS
               set date_last_attempt = sysdate,
                   login_attempts    = v_nCurrIPAttempts
             where client_ip = pi_vSessionClientIP;
            commit;
          
            --check and lock if necessary
            if v_nCurrIPAttempts >= v_nMAX_IP_LOGIN_ATTEMPTS then
            
              update tbicds.FX_IPLOGIN_ATTEMPTS
                 set is_locked = 1
               where client_ip = pi_vSessionClientIP;
              commit;
            
              open rs
             for select (select 0 from dual) as fx_user_id,
                        (select '' from dual) as question_1,
                        (select '' from dual) as question_2,
                        (select 0 from dual) as is_locked,
                        (select 0 from dual) as is_locked
                   from dual;
                   
              po_nStatusCode    := 6;
              po_vStatusComment := 'Invalid username, IP Address Locked!';
   
            end if;
          
          else
          
            --first bad ip attempt
            insert into tbicds.FX_IPLOGIN_ATTEMPTS
              (client_ip,
               login_attempts,
               date_last_attempt,
               is_locked,
               lock_period)
            values
              (pi_vSessionClientIP, 1, sysdate, 0, v_nMAX_IP_LOGIN_TIMEOUT);
            commit;
          
          end if;
      end if;
    
      --open recordset
      open rs for
        select v_nFXUserID as fx_user_id,
               (select question
                  from tbicds.stat_sec_questions q
                 where q.question_id in
                       (select sq.question_id_1
                          from tbicds.fx_sec_questions sq
                         where fx_user_id = v_nFXUserID)) as question_1,
               
               (select question
                  from tbicds.stat_sec_questions q
                 where q.question_id in
                       (select sq.question_id_2
                          from tbicds.fx_sec_questions sq
                         where fx_user_id = v_nFXUserID)) as question_2,
               
               (select t.is_locked
                  from tbicds.fx_user t
                 where t.fx_user_id = v_nFXUserID) as is_locked,
                 
                 (select v_nIPIsLocked from dual) as ip_locked
          from dual; 
          
          
    
    exception
      when others then
        po_nStatusCode    := 1;
        po_vStatusComment := 'PCK_FX_SEC.GetUserQuestions(): ';
    end;

  /* updates user's selected security questions */
  procedure UpdateSecQuestions(pi_vSessionID       in varchar2,
                                 pi_vSessionClientIP in varchar2,
                                 pi_nUserID          in number,
                                 
                                 pi_nQuestionID_1 in number,
                                 pi_vAnswer_1     in varchar2,
                                 
                                 pi_nQuestionID_2 in number,
                                 pi_vAnswer_2     in varchar2,
                                 
                                 pi_nQuestionID_3 in number,
                                 pi_vAnswer_3     in varchar2,
                                 
                                 po_nStatusCode    out number,
                                 po_vStatusComment out varchar2) 
                                 
    is
    
      v_bIsUpdate boolean := false;
      v_nCount    number := 0;
    
    begin
    
      po_nStatusCode    := 0; --0 = success
      po_vStatusComment := '';
    
      select count(*)
        into v_nCount
        from tbicds.fx_sec_questions f
       where f.fx_user_id = pi_nUserID;
    
      if v_nCount > 0 then
        v_bIsUpdate := true;
      end if;
    
      if v_bIsUpdate then
        update tbicds.fx_sec_questions q
           set q.question_id_1 = pi_nQuestionID_1,
               q.answer_1      = pi_vAnswer_1,
               q.question_id_2 = pi_nQuestionID_2,
               q.answer_2      = pi_vAnswer_2
        --q.question_id_3 = pi_nQuestionID_3,
        --q.answer_3      = pi_vAnswer_3
         where q.fx_user_id = pi_nUserID;
      else
        insert into tbicds.fx_sec_questions q
          (fx_user_id,
           question_id_1,
           answer_1,
           question_id_2,
           answer_2,
           question_id_3,
           answer_3)
        values
          (pi_nUserID,
           pi_nQuestionID_1,
           pi_vAnswer_1,
           pi_nQuestionID_2,
           pi_vAnswer_2,
           null,
           null);
      end if;
    
      commit;
    
    exception
      when others then
        po_nStatusCode    := 1;
        po_vStatusComment := 'PCK_FX_SEC.UpdateSecQuestions(): ';
    end;

  /* checks answers submited for the challenge questions */
  procedure CheckSecurityQuestions(pi_vSessionID       in varchar2,
                                     pi_vSessionClientIP in varchar2,
                                     pi_nUserID          in number,
                                     
                                     --pi_nQuestionID_1 in number,
                                     pi_vAnswer_1 in varchar2,
                                     
                                     --pi_nQuestionID_2 in number,
                                     pi_vAnswer_2 in varchar2,
                                     
                                     --pi_nQuestionID_3 in number,
                                     pi_vAnswer_3 in varchar2,
                                     
                                     po_nStatusCode    out number,
                                     po_vStatusComment out varchar2) 
     is
    
      v_nValidQ1 number := 0;
      v_nValidQ2 number := 0;
      v_nValidQ3 number := 0;
    
      v_nAttempts number := 0;
    
    begin
    
      po_nStatusCode    := 0; --0 = success
      po_vStatusComment := '';
    
      select count(*)
        into v_nValidQ1
        from tbicds.fx_sec_questions q
       where q.fx_user_id = pi_nUserID
         and q.answer_1 = pi_vAnswer_1;
    
      select count(*)
        into v_nValidQ2
        from tbicds.fx_sec_questions q
       where q.fx_user_id = pi_nUserID
         and q.answer_2 = pi_vAnswer_2;
    
      
      --select count(*)
      --  into v_nValidQ3
      --  from tbicds.fx_sec_questions q
      -- where q.fx_user_id = pi_nUserID
      --   and q.answer_3 = pi_vAnswer_3;
      
      if (v_nValidQ1 + v_nValidQ2 + v_nValidQ3) >= 2 then
      
        update tbicds.fx_sec_questions t
           set t.answer_attempts = 0, t.last_updated = sysdate
         where t.fx_user_id = pi_nUserID;
        commit;
      
        po_nStatusCode    := 0; --0 = success
        po_vStatusComment := '';
      
      else
      
        select t.answer_attempts
          into v_nAttempts
          from tbicds.fx_sec_questions t
         where t.fx_user_id = pi_nUserID;
      
        v_nAttempts := v_nAttempts + 1;
      
        update tbicds.fx_sec_questions t
           set t.answer_attempts = v_nAttempts, t.last_updated = sysdate
         where t.fx_user_id = pi_nUserID;
        commit;
      
        if v_nAttempts >= 3 then
          update tbicds.fx_user t
             set t.is_locked       = 1,
                 t.flogin_attempts = v_nAttempts,
                 t.last_updated    = sysdate
           where t.fx_user_id = pi_nUserID;
          commit;
        
          po_nStatusCode    := 9; --9 = account locked
          po_vStatusComment := 'Your security answer attempts have exceeded that set by the system, so your account has been locked. ';
          po_vStatusComment := po_vStatusComment ||
                               'Please contact the system administrator to reactivate your login. ';
          po_vStatusComment := po_vStatusComment ||
                               'Security answer is invalid.';
          return;
        else
          po_nStatusCode    := 1; --1 = invalid answer
          po_vStatusComment := 'Security answer is invalid.';
          return;
        end if;
      
      end if;
    
    exception
      when others then
        po_nStatusCode    := 1;
        po_vStatusComment := 'PCK_FX_SEC.CheckSecurityQuestions(): ';
    end;

  /* reset account  password */
  procedure ResetPassword(pi_vSessionID       in varchar2,
                          pi_vSessionClientIP in varchar2,
                          pi_nUserID          in number,
                          pi_vKey             in varchar2,
                          pi_nFXUserID        in number,
                          pi_vUserName        in varchar2,
                          pi_vPassword        in varchar2,
                          pi_vCPassword       in varchar2,
                          pi_vCUserName       in varchar2,
                          po_nStatusCode      out number,
                          po_vStatusComment   out varchar2)
  
   is
    v_nCount  number;
    dtSys     date;
    v_currpwd varchar2(4000);
  
  begin
  
    --default status to good
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    v_nCount  := 0;
    dtSys     := sysdate;
    v_currpwd := '';
  
    pck_fx_sec.ValidatePassword(pi_vKey,
                                pi_nFXUserID,
                                pi_vUserName,
                                'NA',
                                pi_vPassword,
                                'NA',
                                pi_vCPassword,
                                pi_vCUserName,
                                1,
                                po_nStatusCode,
                                po_vStatusComment);
    if po_nStatusCode != 0 then
      return;
    end if;
  
    --update date_password_changed if the user changed the pwd
    select password
      into v_currpwd
      from tbicds.fx_user
     where fx_user_id = pi_nFXUserID;
    if v_currpwd != pi_vPassword then
    
      update tbicds.fx_user
         set date_password_changed = dtSys,
             last_updated          = sysdate,
             last_updated_by       = pi_nFXUserID
       where fx_user_id = pi_nFXUserID;
      commit;
    
    end if;
  
    update tbicds.fx_user
       set is_locked       = 0,
           is_inactive     = 0,
           date_modified   = dtSys,
           updated_by      = pi_nFXUserID,
           reset_password  = 0,
           password        = pi_vPassword,
           last_updated    = sysdate,
           last_updated_by = pi_nFXUserID
     where fx_user_id = pi_nFXUserID;
    commit;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := '178 - An error occurred while updating a patients password, Please contact your system administrator.';
  end;

  /* Gets security questions */
  procedure GetSecurityQuestions(pi_vSessionID       in varchar2,
                                 pi_vSessionClientIP in varchar2,
                                 pi_nUserID          in number,
                                 pi_nQuestionGrp     in number,
                                 po_nStatusCode      out number,
                                 po_vStatusComment   out varchar2,
                                 rs                  out RetRefCursor) 
  is
  
  begin
  
    po_nStatusCode    := 0; --0 = success
    po_vStatusComment := '';
  
    --open recordset
    open rs for
      select *
        from stat_sec_questions q
       where q.active = 1
         and q.question_group = pi_nQuestionGrp;
  
  exception
    when others then
      po_nStatusCode    := 1;
      po_vStatusComment := 'PCK_FX_SEC.GetSecurityQuestions(): ';
  end;

end;
/

